#ifndef __xbee_h
#define __xbee_h

#include "msg_queues.h"

#define XBEE_BUFFER_SIZE MSGLEN-20
#define XBEE_START_DELIMITER 0x7E

// Expected 'next' state
#define XBEE_STATE_READ_START               10
#define XBEE_STATE_READ_LENGTH_HIGH         11
#define XBEE_STATE_READ_LENGTH_LOW          12
#define XBEE_STATE_READ_FRAME_DATA          14
#define XBEE_STATE_READ_CHECKSUM            15

// Command Frame Type
#define XBEE_TX_AT_COMMAND                  0x08
#define XBEE_TX_AT_COMMAND_QUEUE            0x09
#define XBEE_RX_AT_COMMAND_RESPONSE         0x88

#define XBEE_TX_DATA_PACKET                 0x10
#define XBEE_RX_DATA_PACKET                 0x90
#define XBEE_RX_DATA_TX_STATUS              0x8B
#define XBEE_RX_IO_DATA_SAMPLE              0x92
#define XBEE_TX_EXPLICIT_COMMAND            0x11
#define XBEE_RX_EXPLICIT_COMMAND            0x91

#define XBEE_TX_REMOTE_AT_COMMAND           0x17
#define XBEE_RX_REMOTE_AT_COMMAND_RESPONSE  0x97

#define XBEE_TX_CREATE_SOURCE_ROUTE         0x21
#define XBEE_RX_ROUTE_RECORD                0xA1
#define XBEE_RX_NODE_IDENTIFICATION         0x95
#define XBEE_RX_FRAME_MODEM_STATUS          0x8A

typedef struct {
    union {
        unsigned long long_value;
        unsigned char char_value[4]; // Little Endian!!
    } UPPER_32;
    union {
        unsigned long long_value;
        unsigned char char_value[4]; // Little Endian!!
    } LOWER_32;
} XBEE_ADDRESS_64;

typedef struct {
    union {
        unsigned int int_value;
        unsigned char char_value[2]; // Little Endian!!
    } INT_16;
} XBEE_ADDRESS_16;

// Unique Frame Components
typedef struct {
    unsigned char frame_type;
    unsigned char frame_id;
    unsigned char command[2];
    unsigned char data[XBEE_BUFFER_SIZE];
} XBEE_TX_AT_COMMAND_FRAME;
#define XBEE_TX_AT_COMMAND_FRAME_SIZE 4

typedef struct {
    unsigned char frame_type;
    unsigned char frame_id;
    unsigned char command[2];
    unsigned char data[XBEE_BUFFER_SIZE];
} XBEE_TX_AT_COMMAND_QUEUE_FRAME;
#define XBEE_TX_AT_COMMAND_QUEUE_FRAME_SIZE 4

typedef struct {
    unsigned char frame_type;
    unsigned char frame_id;
    unsigned char command[2];
    unsigned char command_status;
    unsigned char data[XBEE_BUFFER_SIZE];
} XBEE_RX_AT_COMMAND_RESPONSE_FRAME;
#define XBEE_RX_AT_COMMAND_RESPONSE_FRAME_SIZE 5

typedef struct {
    unsigned char frame_type;
    unsigned char frame_id;
    XBEE_ADDRESS_64 destination_64;
    XBEE_ADDRESS_16 destination_16;
    unsigned char broadcast_radius;
    unsigned char options;
    unsigned char data[XBEE_BUFFER_SIZE];
} XBEE_TX_DATA_PACKET_FRAME;
#define XBEE_TX_DATA_PACKET_FRAME_SIZE 14

typedef struct {
    unsigned char frame_type;
    XBEE_ADDRESS_64 source_64;
    XBEE_ADDRESS_16 source_16;
    unsigned char recieve_options;
    unsigned char data[XBEE_BUFFER_SIZE];
} XBEE_RX_DATA_PACKET_FRAME;
#define XBEE_RX_DATA_PACKET_FRAME_SIZE 12

typedef struct {
    unsigned char frame_type;
    unsigned char frame_id;
    XBEE_ADDRESS_16 destination_16;
    unsigned char transmit_retry_count;
    unsigned char delivery_status;
    unsigned char discovery_status;
} XBEE_RX_DATA_TX_STATUS_FRAME;
#define XBEE_RX_DATA_TX_STATUS_FRAME_SIZE 7

typedef struct {
    unsigned char frame_type;
    XBEE_ADDRESS_64 source_64;
    XBEE_ADDRESS_16 source_16;
    unsigned char recieve_options;
    unsigned char number_of_samples;
    unsigned char digital_ch_mask[2];
    unsigned char analog_ch_mask;
    unsigned char digital_samples[2];
    unsigned char analog_samples[8];
} XBEE_RX_IO_DATA_SAMPLE_FRAME;
#define XBEE_RX_IO_DATA_SAMPLE_FRAME_SIZE 26

typedef struct {
    unsigned char frame_type;
    unsigned char frame_id;
    XBEE_ADDRESS_64 destination_64;
    XBEE_ADDRESS_16 destination_16;
    unsigned char source_endpoint;
    unsigned char destination_endpoint;
    unsigned char cluster_id[2];
    unsigned char profile_id[2];
    unsigned char broadcast_radius;
    unsigned char transmit_options;
    unsigned char data[XBEE_BUFFER_SIZE];
} XBEE_TX_EXPLICIT_COMMAND_FRAME;
#define XBEE_TX_EXPLICIT_COMMAND_FRAME_SIZE 20

typedef struct {
    unsigned char frame_type;
    XBEE_ADDRESS_64 source_64;
    XBEE_ADDRESS_16 source_16;
    unsigned char source_endpoint;
    unsigned char destination_endpoint;
    unsigned char cluster_id[2];
    unsigned char profile_id[2];
    unsigned char recieve_options;
    unsigned char data[XBEE_BUFFER_SIZE];
} XBEE_RX_EXPLICIT_COMMAND_FRAME;
#define XBEE_RX_EXPLICIT_COMMAND_FRAME_SIZE 18

typedef struct {
    unsigned char frame_type;
    unsigned char frame_id;
    XBEE_ADDRESS_64 destination_64;
    XBEE_ADDRESS_16 destination_16;
    unsigned char remote_options;
    unsigned char command[2];
    unsigned char data[XBEE_BUFFER_SIZE];
} XBEE_TX_REMOTE_AT_COMMAND_FRAME;
#define XBEE_TX_REMOTE_AT_COMMAND_FRAME_SIZE 15

typedef struct {
    unsigned char frame_type;
    unsigned char frame_id;
    XBEE_ADDRESS_64 source_64;
    XBEE_ADDRESS_16 source_16;
    unsigned char command[2];
    unsigned char command_status;
    unsigned char command_data[4];
} XBEE_RX_REMOTE_AT_COMMAND_FRAME;
#define XBEE_RX_REMOTE_AT_COMMAND_FRAME_SIZE 19

typedef struct {
    unsigned char frame_type;
    unsigned char frame_id;
    XBEE_ADDRESS_64 destination_64;
    XBEE_ADDRESS_16 destination_16;
    unsigned char route_options;
    unsigned char num_of_addresses;
    unsigned char addresses[XBEE_BUFFER_SIZE];
} XBEE_TX_CREATE_SOURCE_ROUTE_FRAME;
#define XBEE_TX_CREATE_SOURCE_ROUTE_FRAME_SIZE 14

typedef struct {
    unsigned char frame_type;
    XBEE_ADDRESS_64 source_64;
    XBEE_ADDRESS_16 source_16;
    unsigned char recieve_options;
    unsigned char num_of_addresses;
    unsigned char addresses[XBEE_BUFFER_SIZE];
} XBEE_RX_ROUTE_RECORD_FRAME;
#define XBEE_RX_ROUTE_RECORD_FRAME_SIZE 13

typedef struct {
    unsigned char frame_type;
    XBEE_ADDRESS_64 source_64;
    XBEE_ADDRESS_16 source_16;
    unsigned char recieve_options;
    XBEE_ADDRESS_16 remote_16;
    XBEE_ADDRESS_64 remote_64;
    unsigned char NI_string[2];
    XBEE_ADDRESS_16 parent_16;
    unsigned char device_type;
    unsigned char source_event;
    unsigned char profile_id[2];
    unsigned char manufacturer_id[2];
} XBEE_RX_NODE_IDENTIFICATION_INDICATOR_FRAME;
#define XBEE_RX_NODE_IDENTIFICATION_INDICATOR_FRAME_SIZE 32

typedef struct {
    unsigned char frame_type;
    unsigned char status;
} XBEE_RX_MODEM_STATUS_FRAME;
#define XBEE_RX_MODEM_STATUS_FRAME_SIZE 2

typedef union __XBEE_FRAME {
    XBEE_TX_AT_COMMAND_FRAME            TX_AT_COMMAND;
    XBEE_TX_AT_COMMAND_QUEUE_FRAME      TX_AT_COMMAND_QUEUE;
    XBEE_RX_AT_COMMAND_RESPONSE_FRAME   RX_AT_COMMAND_RESPONSE;
    XBEE_TX_DATA_PACKET_FRAME           TX_DATA_PACKET;
    XBEE_RX_DATA_PACKET_FRAME           RX_DATA_PACKET;
    XBEE_RX_DATA_TX_STATUS_FRAME        RX_DATA_TX_STATUS;
    XBEE_RX_IO_DATA_SAMPLE_FRAME        RX_IO_DATA_SAMPLE;
    XBEE_TX_EXPLICIT_COMMAND_FRAME      TX_EXPLICIT_COMMAND;
    XBEE_RX_EXPLICIT_COMMAND_FRAME      RX_EXPLICIT_COMMAND;
    XBEE_TX_REMOTE_AT_COMMAND_FRAME     TX_REMOTE_AT_COMMAND;
    XBEE_RX_REMOTE_AT_COMMAND_FRAME     RX_REMOTE_AT_COMMAND;
    XBEE_TX_CREATE_SOURCE_ROUTE_FRAME   TX_CREATE_SOURCE_ROUTE;
    XBEE_RX_ROUTE_RECORD_FRAME          RX_ROUTE_RECORD;
    XBEE_RX_NODE_IDENTIFICATION_INDICATOR_FRAME     RX_NODE_IDENTIFICATION;
    XBEE_RX_MODEM_STATUS_FRAME          RX_MODEM_STATUS;
} XBEE_FRAME;

// Common Frame Components
typedef struct __XBEE_FULL_FRAME {
    unsigned char start_delimiter;
    XBEE_ADDRESS_16 length;
    XBEE_FRAME *FRAME;
} XBEE_FULL_FRAME; // 5 bytes overhead

// Overall Data Structure
typedef struct __xbee_data {
    XBEE_FULL_FRAME rcv_frame;
    unsigned char dataind;
    unsigned char checksum_sum;
    unsigned char read_state;
    unsigned char msgtype;
} XBEE_DATA; // 9 bytes overhead


void xbee_init(XBEE_DATA *);
void xbee_read_serial(unsigned char);
void xbee_process_recieved_frame(void);
//void xbee_process_transmit_frame_interrupt(void);
void xbee_process_transmit_frame(void *data, unsigned char length);

void xbee_set_RTS(unsigned char);
unsigned char xbee_read_CTS(void);
void xbee_sleep(void);
void xbee_wake(void);

void ConvertEndian64(XBEE_ADDRESS_64 *src);
void ConvertEndian16(XBEE_ADDRESS_16 *src);

#endif